<?php
###########################################################################
#  Purpose:	This is Image Processor core script. It contains the Image Processor class 
#  File:		ss_image.php
#	Author:		Yuriy Horobey, SmiledSoft.com
#	Copyright:	Yuriy Horobey, SmiledSoft.com
#	Contact:	http://smiledsoft.com/
#	
#	Please do not remove this notice.
#	
#--------------------------------------------------------
# 2005.02.05 Oliver Georgi
# - function set_parameter(JPEG_QUALITY, USE_GD2, USE_COLOR_NAMES) implemented
# - $cfg is no longer a global var - and no array anymore
# - ss_image.config.php is obsolete with this too
# - all $cfg var are $this->var
#
###########################################################################

/**
The core class. Contains most of the functionality of the script.
The freeware functions will be placed in other files and the interface will be provided from the ss_image class
*/
class ss_image{
	/**
	contains resource identifier of the original image
	@type resource
	@see ss_image#commit
	@see ss_image#$img_changed
	*/
	var $img_original;
	/**
	contains resource identifier of the changed image. 
	For example after resizing operation it will contain the resource identifier of the resized image while $img_original will still contain the original image, so there is possibility for undo.
	@type resource
	@see ss_image#commit
	@see ss_image#$img_original
	*/
	var $img_changed;
	
	/**
		The file format of the original image. 
		Has only meaning when created from file. This field is integer, see getimagesize()
		@type integer
		@see ss_image#output
	*/
	var $img_original_type; 
	
	// 2005.02.05 -start-
	// all "global $cfg" -> "//$global $cfg"
	// $cfg[] replaced by following "real" var

	/**	Quality of JPEG output 0-lowest 100-thebest
		@type integer
		@see ss_image#set_parameter
		@see ss_image#output
	*/
	var $cfg_JPEG_QUALITY = 80;

	/**	Determines if GD2 library should be used.
		There are some new functions introduced in GD2 like imagecreatetruecolor() or imagecopyresampled()
		Set this parameter to 'true' if you have GD2 installed and want to have  better quality pictures
		@type boolean
		@see ss_image#set_parameter
		@see ss_image#output
	*/
	var $cfg_USE_GD2 = true;

	/**	Allows to use standard color names along with color codes.
		If this is set to true you can use "black" instead of "#000000" or "white" instead of "#FFFFFF"
		For more color names see file "ss_image.colortohex.php"
		@type boolean
		@see ss_image#set_parameter
		@see ss_image#get_color
	*/
	var $cfg_USE_COLOR_NAMES = true;
	
	/**	Contains RGB of the "transparent color".
		This is used for transparent GIFs and PNG images. The field is set at the stage of creating the class. If this field was set then we can only work with palleted image which may impact the quality of transformed image.
		@type array
		@see ss_image#createimage
		@see ss_image#ss_image
	*/
	var $trans_rgb = array();
	

	///////////////////////////// METHODS ///////////////////////////////////////////
	/**
	Sets config parameters.
	
	@param integer jpeg_quality Quality of the JPEG output 0-the worst 100-the best, default is 80
	@param boolean use_gd2 Allows use of GD2, default is true
	@param boolean use_color_names Allows using color names along with codes, default is true

	@see ss_image#$cfg_JPEG_QUALITY
	@see ss_image#$cfg_USE_GD2
	@see ss_image#$cfg_USE_COLOR_NAMES

	*/
	function set_parameter($jpeg_quality=80, $use_gd2=true, $use_color_names=true) {
		$this->cfg_JPEG_QUALITY		= $jpeg_quality;
		$this->cfg_USE_GD2			= $use_gd2;
		$this->cfg_USE_COLOR_NAMES	= $use_color_names;		
	}
	

	/** The constructor.
	Constructs the object and creates the original image
	@param string src Source of the image. Currently can be file or string use string if your image comes from the database or is generated by other script
	@param string src_kind Determines what is src parameter and can be one of the following
	<ul>
		<li>"f" -- $src is path to file</li>
		<li>"u" -- uploaded file, in this case $src is name of form field &lt;input name='HERE' type='file' ...&gt;</li>
		<li>"s" -- $src is a string containing valid image stream. For example it can be taken from the database blob field</li>
		<li>"s64"--$src is base64 encoded
	</ul>
	*/
	function ss_image( $src, $src_kind="f"){
		global $_FILES;
		$this->img_original_type='';
		switch(strtoupper(trim($src_kind))){
			case "S64":
			//base 64 encoding -- decode and proceed to create from string
			$src = base64_decode($src);
			case "S":
			//create from string

			$this->img_original = imagecreatefromstring($src);

			break;
			case "U":
			$src=$_FILES[$src]['tmp_name'];

			case "F":
			default:
			//create from file. this is also the default behaviour
			$img_info=getimagesize($src);
			//to prevent divide by zero error in resizing
			$this->img_original_type=$img_info[2];
				switch($img_info[2]){//2nd index contains type information
					case "1"://GIF supported was canceled since GD1.6 and is inspecting
						//to be back in middle of 2004
						$this->img_original = imagecreatefromgif($src);
					
					break;
					
					case "2"://JPEG
						$this->img_original = imagecreatefromjpeg($src);
					break;
					
					case "3"://PNG
						$this->img_original = imagecreatefrompng($src);
					break;
					
					case "15"://WBMP
						$this->img_original = imagecreatefromwbmp($src);
					break;
					
					case "16"://XBM
						$this->img_original = imagecreatefromxbm($src);
					break;
					
					default://well format for which we dont have imagecreatefromXX() function
						//lets try create from GD? most lieklly it will fail,
						//but what to do?

						$this->img_original = imagecreatefromGD2($src);

					break;

				}
			break;
		}
		if(!$this->img_original) {
			return null;
		}else{
			$transparency_color_index = imagecolortransparent($this->img_original);


			if($transparency_color_index>-1){
				$this->trans_rgb=@imagecolorsforindex($this->img_original,$transparency_color_index);

			}else{
				$this->trans_rgb=array();
			}
		}
	}

/**
Outputs the image.
This function can output original or changed image either to filesystem (the folder should have write permissions!) or directly to the browser. When outputting to the browser, proper image headers are sent.
@param string where Specify here file name or full path where to output, or
	leave blank to output directly to browser. When saving image to the filesystem please make sure that you have write permissions. The default is direct output to the browser.
@param char what Which image should be output: original or changed?
	<ul><li>"o"-original</li><li>"c"-changed</li></ul>
@param mixed method Specifies format of the output:
<ul>
<li>1 or "GIF" -- GIF format</li>
<li>3 or "PNG" -- PNG format</li>
<li>2 or "JPEG" or "JPG" or left blank-- JPEG format</li>
<li>15 or "WBMP" -- WBMP format</li>
<li>16 or "XBM" -- XBM format</li>
</ul>
@see ss_image#get_supported_types
*/	
	function output($where="", $what="c", $method=""){
		
		$method = strtoupper(trim($method));
		if(!$method){
			$method=$this->img_original_type;
		}

		$what	= strtoupper(trim($what));
		//what to output? original or changed?
		if($what=="O") {
			$image = $this->img_original;
		} else {
			$image = $this->img_changed;
		}

		switch($method){
			case "1":
			case "GIF":
			
			if(!$where){
				header("Content-Type: image/gif");
				imagegif($image);
			}else{
				imagegif($image, $where);
			}
			break;
			case "3"://PNG
			case "PNG":
			@imagesavealpha($imgage, true);
			if(!$where){
				header("Content-Type: image/png");
				imagepng($image);
			} else {
				imagepng($image, $where);
			}
			break;
			//Note: WBMP support is only available if PHP was compiled against GD-1.8 or later
			case "15"://WBMP
			case "WBMP":
			if(!$where){
				header("Content-Type: image/wbmp");
				imagewbmp($image);
			} else {
				imagewbmp($image, $where);
			}
			break;
			case "16"://XBM
			case "XBM":
			/*
			Warning  {from manual!}
			This function is currently not documented; only the argument list is available.
			Note: This function is only available if PHP is compiled with the bundled version of the GD library.

			*/
			if(!$where){
				header("Content-Type: image/xbm");
				imagexbm($image);
			} else {
				imagexbm($image, $where);
			}
			break;
			case "2":
			case "JPEG":
			case "JPG":
			default:
			if(!$where) {
				header("Content-Type: image/jpeg");
				imagejpeg($image, "", $this->cfg_JPEG_QUALITY);
			} else {
				imagejpeg($image, $where, $this->cfg_JPEG_QUALITY);
			}
			break;
		}


	}

/**
 Returns string with types supported by GD.
 This function is simple decifering imagetypes() and returns string consiting of abbreviations of the image types separated by blanks.<br> 
 Like this: "PNG GIF JPG JPEG WBMP XPM";
 @return string
*/

	function get_supported_types(){
		$bits = imagetypes();
		$res = ""; // variable for result
		if($bits & IMG_PNG)		$res .= "PNG ";
		if($bits & IMG_GIF)		$res .= "GIF ";
		if($bits & IMG_JPG)		$res .= "JPG ";
		if($bits & IMG_JPEG)	$res .= "JPEG ";
		if($bits & IMG_WBMP)	$res .= "WBMP ";
		if($bits & IMG_XPM)		$res .= "XPM ";
		$res = trim($res);
		return $res;

	}
/** Commits changes made to the image.
 Destroys the original image and copies resource identifier of the changed image to the original image. 
 There is no undo for this operation. 
 @see ss_image::$img_original
 @see ss_image::$img_changed
*/
	function commit(){
		//copy changed image to original so all next transformation will be
		// based on already made ones.
		@imagedestroy($this->img_original);
		$this->img_original = $this->img_changed;
		$this->img_changed=null;

	}

/**
Cleans up the memory.
*/
	function destroy(){
		@imagedestroy($this->img_original);
		@imagedestroy($this->img_changed);
	}

/**
Returns width of the image.
@param char which determines the image which width is to be returned: 
<ul>
	<li>"o" original image, default value</li>
	<li>"c" changed image</li>
</ul>
@return integer
*/
	function get_w(	$which="o" ){ 
		return trim(strtoupper($which))=="O" ? imagesx($this->img_original) : imagesx($this->img_changed);
	}

/**
Returns height of the image.
@param char which determines the image which height is to be returned: 
<ul>
	<li>"o" original image, default value</li>
	<li>"c" changed image</li>
</ul>
@return integer
*/
	function get_h(	$which="o" ){
		return trim(strtoupper($which))=="O" ? imagesy($this->img_original) : imagesy($this->img_changed);
	}



/**
Creates image.
createimage method creates image resource using GD 1.x.x or GD 2.x.x depending on configuration parameters.
@return resource
@see ss_image#set_parameter
@see ss_image#$trans_rgb
@see ss_image#$cfg_USE_GD2
*/
	function createimage($width, $height){
		
		if($this->trans_rgb){
			$image = imagecreate($width,$height);
			$trans_clr = imagecolorallocate($image, $this->trans_rgb['red'], $this->trans_rgb['green'], $this->trans_rgb['blue']);
			imagefilledrectangle($image,0,0,$width,$height,$trans_clr);
			imagecolortransparent($image,$trans_clr);
		}else{
			$image = $this->cfg_USE_GD2 ? imagecreatetruecolor($width,$height) : imagecreate($width,$height);
		}
		return $image;
	}

/** Sets color of pixel under X and Y coordinates as transparent.
@param integer x horizontal coordinate
@param integer y vertical coordinate
@return integer The identifier of the new (or current, if none is specified) transparent color is returned. 
*/
	//this will set transparent color to whatewer color is pixel with given x, y
	function set_transparentcolor($x=0,$y=0){
		$color=imagecolorat($this->img_original,$x,$y);
		return imagecolortransparent($this->img_original,$color);
	}

/** Allocates specified color in specified image
@param string color The color to be allocated. It should be in hex form like FF00FF without leading #, just RRGGBB. <br>
if cfg_USE_COLOR_NAMES is true you may specify color by its standard web name like "olivedrab" instead of 6B8E23.
@param char dest specifies the image where to allocate the color, it can be
<ul>
	<li>'O' or 'o'-- original image</li>
	<li>'C' or 'c' -- changed image</li>
</ul>
 @see ss_image::$img_original
 @see ss_image::$img_changed
 @return integer returns a color identifier the same as would imagecolorallocate() return
*/
	function get_color($color,	$dest="c" ){
		if($this->cfg_USE_COLOR_NAMES){
			//translate color name to RGB hex string
			require_once(dirname(__FILE__)."/ss_image.colortohex.php");
			$color=colortohex($color);
		}

		// we are not going to check this
		$r = hexdec("0x".substr($color, 0,2));
		$g = hexdec("0x".substr($color, 2,2));
		$b = hexdec("0x".substr($color, 4,2));

		if(strtoupper(trim($dest))=="O") {
			$img = $this->img_original;
		} else {
			$img = $this->img_changed;
		}
		$c = imagecolorallocate($img, $r, $g, $b);

		return $c;

	}

	############################################################
	#
	#  These are Image Processing functions
	#  They are independent one from each other so you can delete those which
	#  you are not going to use to save space
	#


	/**
	Resizes the image.
	The $img_origianl should contain valid image resource identificator before calling this function.
	The result will be stored in the $img_changed
	@param mixed desired_width width of the new bounding rectangle
	@param mixed desired_height height of the new bounding rectangle<br>
	<blockquote>If one of these parameters is set to '*' that means that we dont care about mode and result image will be 
	set to fit the other paramter. For example if desired_width is "*" and desired_height is "108" then the result image will be 108 of height no matter which will be the width, W to H ratio will remain in rare case when both of the parameters are "*" then the iamge will be just copied to $img_changed</blockquote>
	@param char mode determines how should the image be resized. 
	<br>The possible values are:
	<ul>
	<li>'-' -- Resizes image in such a way to fit into rectangle given by desired_width and desired height. X to Y ratio is kept.</li>
	<li>'+' -- Resizes image in such a way to cover rectangle given by desired_width and desired height. X to Y ratio is kept.</li>
	<li>'--' Same as '-' mode but resizing will occure only if original image covers the given rectangle.</li>
	<li>'++' Same as '+' mode but resizing will occure only if original image is smaller than the given rectangle.</li>
	<li>'e' or '0' (zero) -- Resizes exactly to dimensions given by desired_width and desired_height. If one of these parameters is set to '*' the corresponding dimension will be not changed. X to Y ratio is not kept!</li>
	</ul>
	@return boolean
	@see ss_image::$img_original 
	@see ss_image::$img_changed
	*/
	function set_size(	$desired_width='*',	$desired_height='*', $mode="-"	){
		
		//prepare parameteres
		$res = false;
		$new_img = NULL;
		$mode=strtoupper(trim($mode));
		$desired_width  = trim($desired_width);
		$desired_height = trim($desired_height);

		$org_w = $this->get_w();
		$org_h = $this->get_h();


		if( $mode=="--" ){
			if (($org_w <= $desired_width ) and ($org_h <= $desired_height ) ){
				// image is already smaller than the given region do not resize.

				$desired_width="*";
				$desired_height="*";


			} else {
				$mode ="-";
			}

		}


		if( $mode=="++" ){
			if (($org_w >= $desired_width ) and ($org_h >= $desired_height ) ){
				// image is already larger than the given region do not resize.

				$desired_width="*";
				$desired_height="*";


			} else {
				$mode ="+";
			}

		}


		//calculate new sizes:
		if($desired_width=="*" and $desired_height=="*"){
			//no resizing just copy parameters:
			$this->img_changed = $this->createimage($this->get_w(), $this->get_h());
			return imagecopy($this->img_changed, $this->img_original, 0, 0, 0, 0, $this->get_w(), $this->get_h());

		} else {

			if(!$org_h) {
				$org_h=1; // prevent divide by zero
			}

			$ratio = $org_w / $org_h;


			if($desired_width=="*"){
				//we dont care what will be new width, image must be of given height
				//and width should be according to the ration
				$desired_height = (int)$desired_height;
				$new_w = $desired_height*$ratio;
				$new_h = $desired_height;
			}elseif($desired_height=="*"){
				//we dont care what will be new height, image must be of given width
				//and height should be according to the ration

				$desired_width  = (int)$desired_width;
				$new_w = $desired_width;
				$new_h = $desired_width/$ratio;
			}else{
				// the desired width and height are given as well as resizing mode
				// lets calculate new width and height..
				$desired_width  = (int)$desired_width;
				$desired_height = (int)$desired_height;
				if(!$desired_width)	$desired_width=1;
				if(!$desired_height)$desired_height=1;

				switch($mode){
					case "0":// compatibility with hft_image 5.x series
					case "E"://resize to fit exactly
					$new_w=$desired_width;
					$new_h=$desired_height;
					break;
					case "+"://overlap given region
					//suppose $new_w will be $desired_width
					$new_h = $desired_width / $ratio;
					if($new_h>=$desired_height) $new_w=$desired_width;
					else{//wrong idea.. right idea was new_h=$desired_height
					$new_w=$ratio*$desired_height;
					$new_h=$desired_height;
					}
					break;
					case "-": //fit into given region
					default:
					//suppose $new_w will be $desired_width
					$new_h = $desired_width / $ratio;
					if($new_h<=$desired_height) $new_w=$desired_width;
					else{//wrong idea.. right idea was new_h=$desired_height
					$new_w=$ratio*$desired_height;
					$new_h=$desired_height;
					}

					break;
				}
			}
			$new_img = $this->createimage($new_w,$new_h);

			if($this->cfg_USE_GD2){
				//use GD2 -- better quality, but not all server have proper installation of it.


				$res= imagecopyresampled(

				$new_img, // destination
				$this->img_original, //source
				0,0,// destination coords
				0,0,//source coords
				$new_w,//new width
				$new_h,//new height
				$org_w,
				$org_h
				);


			} else {
				//worse quality but better compatibility


				$res = imagecopyresized(
				$new_img, // destination
				$this->img_original, //source
				0,0,// destination coords
				0,0,//source coords
				$new_w,//new width
				$new_h,//new height
				$org_w,
				$org_h
				);

			}
		}

		if($res){
			$this->img_changed = $new_img;
		}
		return $res;

	}

/** Rotates the image.
This fucntion will rotate image $img_origianal and fill the zones not covered by the result image with given background color.
@param integer a angle in degrees 0..359
@param string  bkcolor the code for background color. If the cfg_USE_COLOR_NAMES is 'true' then you can also use standard color names like 'blue' or 'white'.<br>
The result image is stored to $img_changed
@see ss_image::$img_original 
@see ss_image::$img_changed
@see ss_image::$cfg_USE_COLOR_NAMES

@return void

*/

	function rotate(
			$a, //angle in degrees 0-359,999..99
			$bkcolor // background color for zone not covered with rotation
					// in html format 'RRGGBB'	-- Hex digits like 'FF0000' - red
					// 'FFFFFF' -- white etc or use word description like red green white 
					// blue black .. see below
		){
		global $cfg;
		//process color

		$c=$this->get_color($bkcolor,"o");

		$this->img_changed = imagerotate($this->img_original, $a, $c);

	}

/**
Converts the image to greyscale.
The result image is stored to $img_changed
@see ss_image::$img_original 
@see ss_image::$img_changed
@return void
*/
	function gray(){ 
		@imagedestroy($this->img_changed);
	
		$this->img_changed=imagecreate($this->get_w(), $this->get_h()); //not true color!!
		//it won't produce grayscale!!
		
		//allocate gray scale
		for ($c = 0; $c < 256; $c++) {
			ImageColorAllocate($this->img_changed, $c,$c,$c);
		}
	
	
		imagecopymerge(	$this->img_changed,
			$this->img_original,
			0,0,//destination x y
			0,0,//source x y
			$this->get_w(), //source sizes
			$this->get_h(),
			100//no alpha -- 100% copy
		);
	
	}

/**
Puts a watermark over the image.
A watermark is another image which will be visible over the original as semitransparent

@param string src the source of the watermark image. This parameter is the same as src in the constructor
@param integer transparency determines how transparent should be the watermark 0-invisible, fully transparent 100-opaque
@param integer x ..
@param integer y position on the original image where top left corner of the watermark image should be placed
@param boolean makegray if set to true will convert the watermark to the greyscale before applying to the original image
@param string src_kind Determines what is src parameter and can be one of the following
	<ul>
		<li>"f" -- $src is path to file</li>
		<li>"u" -- uploaded file, in this case $src is name of form field &lt;input name='HERE' type='file' ...&gt;</li>
		<li>"s" -- $src is a string containing valid image stream. For example it can be taken from the database blob field</li>
		<li>"s64"--$src is base64 encoded
	</ul>

@param mixed desired_w see set_size()
@param mixed desired_h see set_size()
@param mixed resizemode see set_size()

The result image is stored to $img_changed
@see ss_image::$img_original 
@see ss_image::$img_changed
@see ss_image::ss_image
@see ss_image::set_size
@see ss_image::gray
@return void
*/


	function watermark(
	$src,	$transparency=75, $x=0,	$y=0, $makegray=false, 
	//rarely used params:
	$src_kind="f", 	$desired_w="*",	$desired_h="*", $resizemode='-'){

		$img_watermark = new ss_image($src, $src_kind);
		if(trim($desired_w) != "*" or trim($desired_h) !="*" ){
			$img_watermark->set_size($desired_w, $desired_h);
			$img_watermark->commit();
		}

		@imagedestroy($this->img_changed);
		$this->img_changed= $this->createimage($this->get_w(),$this->get_h());
		imagecopy($this->img_changed,$this->img_original,0,0,0,0,$this->get_w(),$this->get_h());


		imagealphablending($this->img_changed, true);
		imagealphablending($img_watermark->img_original, true);

		if($makegray){
			$img_watermark->gray();
			$img_watermark->commit();//save changes
		}

		$img_watermark->set_transparentcolor();

		$res= imagecopymerge(	$this->img_changed,
		$img_watermark->img_original,
		$x,$y,//destination x y
		0,0,//source x y
		$img_watermark->get_w(), //source sizes
		$img_watermark->get_h(),
		$transparency
		);



		$img_watermark->destroy();

		return $res;



	}
/**
Crops the image to the given rectangle
@param integer x left coordinate of the rectangle
@param integer y top coordinate of the rectangle
@param integer width width of the rectangle
@param integer height height of the rectangle
@return integer returns the same result as the imagecopy() would
*/
	function crop($x, $y, $width, $height ){

		//lets change the height width of the specified crop region if it is out of original image bounds
		$org_w = $this->get_w();
		$org_h = $this->get_h();

		if($x+$width>$org_w)$width=$org_w-$x;
		if($y+$height>$org_h)$height=$org_h-$y;

		@ImageDestroy($this->img_changed);
		$this->img_changed= $this->createimage($width,$height);

		return imagecopy($this->img_changed, $this->img_original, 0,0,$x,$y, $width, $height);

	}

/** Writes text over the image
This fucntion writes text over the image. It supports True Type Fonts and if they are used then the text can be written in different directions and can be resized.
@param string text the text to be writen
@param integer x ..
@param integer y coordinates of the text.<br>
<b>Note:</b><blockquote>There is different position of base point for builtin and TTF fonts. For the built in (raster) fonts the basepoint is the top-left corner of the character, but for the TTF it is located on the 'baseline'. So the difference is on the Y dimension and usually is something like 80% of the font height. However this is individual for the every font. Just keep in mind this when you're going to write something at the very bottom of the image </blockquote>

@param string color code for the color of the text.  If the cfg_USE_COLOR_NAMES is 'true' then you can also use standard color names like 'blue' or 'white'.
@param mixed font determines what font to use. 
<ul><li>Numerical -- If this parameter is numerical like 1-5 then the built-in or loaded font is used. You can design your own font and load it with GD function imageloadfont(). This function will return you an integer identifier of the font which you can use here. </li><li>String -- If this parameter is string the it is considered to be file name of the TTF font. Then you can also use size and angle parameters</li></ul>

@param real size sets the size the size of TTF font. Not applicable for built in (raster) fonts
@param real angle sets the direction of the text. Is used with TTF fonts only. Not applicable for built in (raster) fonts

<br>
The result image is stored to $img_changed
@see ss_image::$img_original 
@see ss_image::$img_changed
@see ss_image::$cfg_USE_COLOR_NAMES

@return void
*/
	function text($text, $x=0,$y=0, $color="000000", $font=3, $size=12.0,$angle=0 ){
		//prepare changed image
		@imagedestroy($this->img_changed);
		$this->img_changed= $this->createimage($this->get_w(),$this->get_h());

		imagecopy($this->img_changed,$this->img_original,0,0,0,0,$this->get_w(),$this->get_h());
		$color=$this->get_color($color);

		//which writting method to use? TTF or standard?
		if(intval($font)){
			$font=(int)$font;
			return imagestring($this->img_changed,$font,$x,$y,$text,$color);
		}else{
			//this is TTF
			imagettftext ( $this->img_changed, $size, $angle, $x, $y, $color, $font, $text);
		}

	}
/**
Drops shadow.
This function will drop semitransparent shadow filling the background with the given color
@param integer dx ofset by X
@param integer dy ofset by Y
@param integer transparency the transparency of the shadow 0-transparent, invisible; 100-opaque<br>
The shadow is greyscalled
@param string  bkcolor the code for background color. If the cfg_USE_COLOR_NAMES is 'true' then you can also use standard color names like 'blue' or 'white'.<br>
The result image is stored to $img_changed
@see ss_image::$img_original 
@see ss_image::$img_changed
@see ss_image::$cfg_USE_COLOR_NAMES

@return void

*/
	function shadow( $dx=8, $dy=8, $transparency=75, $bkcolor="FFFFFF" ){

		$this->gray();
		$img_gray = $this->img_changed;

		$new_w =$this->get_w()+abs($dx);
		$new_h=$this->get_h()+abs($dy);
		$this->img_changed= $this->createimage($new_w,$new_h);

		$color= $this->get_color($bkcolor);

		imagefilledrectangle($this->img_changed, 0,0, $new_w,$new_h, $color);

		if($dx<0){
			$org_x=-$dx;
			$shd_x = 0;
		}else{
			$org_x=0;
			$shd_x = $dx;
		}
		if($dy<0){
			$org_y=-$dy;
			$shd_y = 0;
		}else{
			$org_y=0;
			$shd_y = $dy;
		}

		imagealphablending($this->img_changed, true);


		//the shadow
		imagecopymerge($this->img_changed,$img_gray,$shd_x,$shd_y,0,0,$this->get_w(),$this->get_h(),$transparency);
		imagedestroy($img_gray);

		//the original
		imagecopy($this->img_changed,$this->img_original,$org_x,$org_y,0,0,$this->get_w(),$this->get_h());


	}


	/** Applies 'Unsharp' filter.
	This function will make image looking sharper. It is especially useful for thumbnails.<br>
	Note: This fucntion comes with ImageProcessor as the freeware, see copyright notice in ss_image.unsharp.php<br>
	@param integer amount
	@param real radius
	@param integer threshold
	@return resource Resource identifier of the new image.
The result image is also stored to $img_changed
@see ss_image::$img_original 
@see ss_image::$img_changed
	*/

	function unsharp($amount=80, $radius=0.5, $threshold=3){
		require_once(dirname(__FILE__)."/ss_image.unsharp.php");
		return $this->img_changed = UnsharpMask($this->img_original, $amount, $radius, $threshold);
	}

/** Applies mask to the image.
Using mask you can higlight the portion of the picture or make custom crop.
The mask should be palette image. One of the collors (identified by X and Y coordinates) will be considered to be transparent. Mask image should be of the same dimensions as the original image. Pixels of the original image which lay under 'transparent' pixels of the mask image will not be change at all, while all other pixels will be mixed with mask pixels according to the transparency parameter.
@param string src the source of the mask, same as in ss_image()
@param integer transparency determines how visible should be nontransparent part of the mask. 0-it is not visible 100-opaque, just copied over the original image. Keep in mind, that the mask already has transparent part which is not affected by this parameter
@param integer transparent_x X coordinate of the pixel of the 'transparent' color
@param integer transparent_y Y coordinate of the pixel of the 'transparent' color<br>

@param char src_kind kind of the source of the mask, same as in ss_image() (determines whether src is file, uploaded file, binary stream etc)

@return integer
*/
	function mask(
	$src,		// source of watermark can be file, string, base64 encoded string
	// see create()
	$transparency=100, // how visible should be mask? 0-100
	//(invisible-fullvisible)

	$transparent_x=0,
	$transparent_y=0, //coordinates of pixel which color will be considered to be "transparent"


	//rare used params:
	$src_kind="f" // same as in create()

	){

		$img_mask = new ss_image($src, $src_kind);

		@imagedestroy($this->img_changed);
		$this->img_changed= $this->createimage($this->get_w(),$this->get_h());
		imagecopy($this->img_changed,$this->img_original,0,0,0,0,$this->get_w(),$this->get_h());


		imagealphablending($this->img_changed, true);
		imagealphablending($img_mask->img_original, true);


		$img_mask->set_transparentcolor($transparent_x,$transparent_y);

		$res= imagecopymerge(	$this->img_changed,
		$img_mask->img_original,
		$x,$y,//destination x y
		0,0,//source x y
		$img_mask->get_w(), //source sizes
		$img_mask->get_h(),
		$transparency
		);

		$img_mask->destroy();

		return $res;

	}

}
//class--------------------------------------------------------------------------------------------------------------------

?>